/*
 * This file is part of the DistSim distributed simulation framework (client)
 * Copyright (C) 2007 Ulf Hermann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package brn.distsim.client;

import java.awt.event.ActionEvent;
import java.sql.SQLException;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JTable;

import brn.distsim.client.data.Group;
import brn.distsim.client.data.Study;
import brn.distsim.client.tableModels.AddRemoveTableModel;
import brn.distsim.client.tableModels.EditableTableModel;
import brn.distsim.client.tableModels.FileInTableModel;
import brn.distsim.client.tableModels.FileOutTableModel;
import brn.distsim.client.tableModels.ParametersTableModel;
import brn.distsim.client.ui.AddRemoveToolBar;
import brn.distsim.client.ui.ClientFrame;
import brn.distsim.client.ui.SimulationPanel;


public class DefinitionsData extends EditableTableModel<Study, Group> {

	private static final long serialVersionUID = 996374744833965943L;

	private FileInTableModel inFiles;

	private FileOutTableModel outFiles;

	private ParametersTableModel parameters;

	public DefinitionsData(ClientFrame parent, Study data) {
		super(parent, data, null);
		if (data == null) return;
		try {
			SimulationPanel panel = frame.getSimulationPanel();
			parameters = new ParametersTableModel(frame, data, panel
					.getParametersTable());
			inFiles = new FileInTableModel(frame, data, panel.getFilesInTable());
			outFiles = new FileOutTableModel(frame, data, panel
					.getFilesOutTable());
			connectTable(panel.getParametersTable(), panel
					.getParametersToolBar(), parameters);
			connectTable(panel.getFilesInTable(), panel.getInFilesToolBar(),
					inFiles);
			connectTable(panel.getFilesOutTable(), panel.getOutFilesToolBar(),
					outFiles);
			data.loadFromDb();
			for (JButton b : panel.getGroupButtons()) {
				reconnectButton(b, this);
			}
			readData();
		} catch (SQLException e) {
			error(e);
		}
	}

	protected boolean examineAction(ActionEvent arg0) throws SQLException {
		if (super.examineAction(arg0))
			return true;
		else {
			String command = arg0.getActionCommand();
			if (command.equals("selectSingle")) {
				Group g = data.get(getSelected());
				g.performDbUpdate();
				g.createSimulations();
				g.commit();
				return true;
			} else {
				return false;
			}
		}
	}
	
	private void connectTable(JTable table, AddRemoveToolBar toolBar,
			AddRemoveTableModel model) {
		table.setModel(model);
		reconnectButton(toolBar.getAddButton(), model);
		reconnectButton(toolBar.getRemoveButton(), model);
	}

	public void fireTableStructureChanged() {
		parameters.fireTableStructureChanged();
		inFiles.fireTableStructureChanged();
		outFiles.fireTableStructureChanged();
	}

	protected Group createData() {
		String groupName = JOptionPane.showInputDialog("input group name");
		if (groupName != null) {
			Group group = new Group(DBConnector.getDefinitions(), groupName,
					data.getId());
			return group;
		} else
			return null;
	}

	protected int getSelected() {
		return frame.getSimulationPanel().getParametersTable()
				.getSelectedColumn() - 1;
	}

	protected void setSelected(int pos) {
		SimulationPanel p = frame.getSimulationPanel();
		p.getParametersTable().setColumnSelectionInterval(pos + 1, pos + 1);
		p.getFilesInTable().setColumnSelectionInterval(pos + 1, pos + 1);
		p.getFilesOutTable().setColumnSelectionInterval(pos + 1, pos + 1);
	}

	@Override
	protected Group cloneData(Group orig) {
		String groupName = JOptionPane.showInputDialog("input group name");
		if (groupName != null)
			return new Group(orig, groupName);
		else
			return null;
	}

	@Override
	protected void connect(Group newData) {
		try {
			for (Group g : data) {
				g.performDbUpdate();
				g.createSimulations();
				g.commit();
			}
			
		} catch (SQLException e) {
			error(e);
		} catch (IllegalArgumentException e) {
			error(e);
		}
	}

	@Override
	public int getColumnCount() {
		// nothing to do in principle, but we can return the size()
		return data.size();
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		// nothing to do, we don't know which table
		return null;
	}

	@Override
	protected void readData() {
		names.clear();
		for (int i = 0; i < data.size(); ++i) {
			names.add(data.get(i).getName());
		}
		parameters.readData();
		inFiles.readData();
		outFiles.readData();
		fireTableStructureChanged();
	}
	
	protected boolean remove(int pos) {
		if (pos < 0)
			return false;
		data.remove(pos);
		names.remove(pos);
		fireTableStructureChanged();
		return true;
	}
}
