/*
 * This file is part of the DistSim distributed simulation framework (client)
 * Copyright (C) 2007 Ulf Hermann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package brn.distsim.client.data;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * A StudyTree represents all studies of the same name in the database. The
 * studies will differ in their versions
 *
 * @author alve
 *
 */

public class StudyTree extends DbBackedData<Study> {

	protected String name;

	/**
	 * defines a StudyTree, providing a name and a database connection nothing
	 * is loaded from the database here.
	 *
	 * @param definitions
	 *            a connection to a database
	 * @param name
	 *            common name of the studies belonging to this tree
	 * @throws SQLException
	 */
	public StudyTree(Connection definitions, String name) throws SQLException {
		super(definitions);
		this.name = name;
	}

	/**
	 * loads this study tree from the database, if its name and database
	 * connection are valid. The tree and all the studies are loaded from the
	 * database, but no groups. loadFromDB always discards previously loaded
	 * dependent objects, in this case studies
	 */
	@Override
	public void loadFromDb() throws SQLException {
		ResultSet studies = statement
				.executeQuery("SELECT * FROM studies WHERE name='" + name
						+ "' ORDER BY version;");
		super.loadFromDb(studies);
	}

	/**
	 * loads a tree, reassigning the name at the same time. Like this one
	 * studytree object can be reused for multiple trees. loadFromDB always
	 * discards previously loaded dependent objects, in this case studies
	 *
	 * @param name
	 *            the name of the studies to be loaded
	 * @throws SQLException
	 *             if an error occurs in the SQL queries
	 */
	public void loadFromDB(String name) throws SQLException {
		this.name = name;
		loadFromDb();
	}

	/**
	 * creates an empty study with this tree's DB connection
	 */
	@Override
	public Study getEmptyDependent() {
		return new Study(definitions, 0);
	}

	/**
	 * creates one study from a row in the given result set
	 */
	@Override
	public Study getDependent(ResultSet data) throws SQLException {
		return new Study(definitions, data);
	}

	/**
	 * clones a study
	 *
	 * @param other
	 *            the study to be cloned
	 * @return the cloned study
	 */
	@Override
	public Study getDependent(Study other) {
		return new Study(other, other.getVersion(), other.getCommand());
	}

	/**
	 * returns a list of unique names of all studies found in the given database
	 *
	 * @param db
	 *            connection to the database to be searched
	 * @return a list of names of studies in the DB
	 * @throws SQLException
	 *             if the query fails
	 */
	public static List<String> getNames(Connection db) throws SQLException {
		ResultSet names = db.createStatement().executeQuery(
				"SELECT DISTINCT name FROM studies ORDER BY name");
		List<String> ret = new ArrayList<String>();
		while (names.next()) {
			ret.add(names.getString("name"));
		}
		return ret;
	}

}
