/*
 * This file is part of the DistSim distributed simulation framework (client)
 * Copyright (C) 2007 Ulf Hermann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package brn.distsim.client.tableModels;

import java.awt.event.ActionEvent;
import java.sql.SQLException;
import java.util.ArrayList;

import javax.swing.JTable;

import brn.distsim.client.data.DbBackedData;
import brn.distsim.client.data.Saveable;
import brn.distsim.client.ui.ClientFrame;


public abstract class EditableTableModel<E extends DbBackedData<D>, D extends Saveable>
		extends AddRemoveTableModel<E> {

	private static final long serialVersionUID = 1080673481662602313L;

	public EditableTableModel(ClientFrame parent, E data, JTable subject) {
		super(parent, data, subject, new ArrayList<String>());
	}

	protected boolean examineAction(ActionEvent arg0) throws SQLException {
		if (super.examineAction(arg0))
			return true;
		else {
			String command = arg0.getActionCommand();
			int pos = getSelected();

			if (command.equals("select")) {
				data.commit();
				if (pos < 0)
					connect(null);
				else
					connect(data.get(pos));
				switchTabs();
			} else if (command.equals("clone")) {
				if (pos < 0)
					return true;
				D newData = cloneData(data.get(pos));
				names.add(newData.getName());
				insertData(newData);
			} else
				return false;
			return true;
		}
	}

	protected String create() {
		D newData = createData();
		if (newData != null) {
			insertData(newData);
			String name = newData.getName();
			names.add(name);
			return name;
		} else
			return null;
	}

	protected boolean remove(int pos) {
		if (pos < 0)
			return false;
		if (!data.isInserted(pos))
			return false;
		data.remove(pos);
		names.remove(pos);
		fireTableStructureChanged();
		return true;
	}

	protected void insertData(D newData) {
		if (newData == null)
			return;
		data.add(newData);
		fireTableStructureChanged();
	}

	

	protected abstract void connect(D newData);

	protected abstract D createData();

	protected abstract D cloneData(D orig);

	public int getRowCount() {
		return data.size();
	}

	public abstract int getColumnCount();

	public abstract Object getValueAt(int rowIndex, int columnIndex);

	@Override
	protected int getSelected() {
		return subject.getSelectedRow();
	}

	@Override
	protected void setSelected(int pos) {
		if (pos < 0)
			return;
		else
			subject.setRowSelectionInterval(pos, pos);
	}

	

	public boolean isCellEditable(int rowIndex, int columnIndex) {
		if (columnIndex > 0 && data.isInserted(columnIndex))
			return true;
		else
			return false;
	}
}
