package jplot;
/*
 * This file is part of jPlot
 *
 * jPlot is a Java version of Tim Shepard's xplot which is distributed 
 * under the M.I.T. license (as below).
 *
 * Tim Shepard's original xplot is an amazing piece of software. It was
 * designed to be extraordinarily fast and efficient and as such, it has
 * only a very thin user interface. Those design goals, however, make
 * xplot extremely difficult to port to new architectures and also make it
 * difficult to use for the casual user. jPlot was designed to address
 * these limitations while keeping as much of the original efficiency as
 * possible. We thank Tim Shepard for his hard work on xplot and hope that
 * jPlot can live up to his high standards.
 *
 * jPlot was developed by Avinash Lakhiani at the Internetworking Research
 * Lab (IRG), Ohio University, as part of Dr. Shawn Ostermann's tcptrace
 * project. The main goal behind the development of jPlot was to make the 
 * plotting tool used with tcptrace more portable. As a result, jPlot
 * retains many of the original algorithms implemented in xplot, and as per
 * the terms and conditions of the M.I.T. license, continue to remain under
 * that license. The M.I.T license is as follows:
 *
 * --- M.I.T License Begin ---
 * 
 * Copyright 1992,1993 by the Massachusetts Institute of Technology.
 *                   All rights reserved.
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
 * OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
 * limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
 * MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
 * OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
 * PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 *
 * The name of the Massachusetts Institute of Technology or M.I.T. may
 * NOT be used in advertising or publicity pertaining to distribution of
 * the software.  Title to copyright in this software and any associated
 * documentation shall at all times remain with M.I.T., and USER agrees
 * to preserve same.
 * 
 * --- M.I.T. License End ---
 * 
 * jPlot also has a lot of added features that are not part of the original
 * xplot code. Those sections of code are distributed under the following
 * BSD License:
 *
 * --- BSD License Begin ---
 *
 * Copyright (c) 2002, 2003 by Avinash Lakhiani
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *	
 *   - Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 *   - Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution. 
 *   - Neither the name of Ohio University nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission. 
 *	   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * --- BSD License End ---
 * 
 * You should have received a file named "DISTINCTION" along with this
 * distribution, that clearly identifies all the algorithms that were
 * retained from the original xplot code, and remain under the M.I.T.
 * license. The rest of the code was written for jPlot, and falls under
 * the BSD license.
 * 
 * jPlot is free software; you can redistribute it and/or modify it under the
 * terms and conditions of both the above licenses.
 *
 * jPlot is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the above licenses for more details.
 *
 * Author:  Avinash Lakhiani
 *          Internetworking Research Group
 *          Ohio University
 *          Athens, OH
 *          avinash.lakhiani@ohiou.edu
 *          http://www.tcptrace.org/jPlot
 *
 */

/*
 * $Id: Arguments.java,v 1.2 2002/12/04 01:58:52 alakhian Exp $
 */ 

/* Import Packages */
import java.lang.Object;
import java.util.ArrayList;

/* Class       - Arguments 
 * Description - holds all the command-line arguments, and provides and interface for
 * their manipulation.
 * Extends     - NULL
 * Implements  - NULL
 */
public class Arguments 
{
   /* The command-line arguments */
   private boolean one;      // display one file at a time
   private boolean margin;   // include an extra margin around the plot
   private boolean help;     // provide help on jPlot
   private boolean version;  // print the version of jPlot and exit
   private boolean tile;     // tile the plot frames
   private boolean cascade;  // cascade the plot frames
   private boolean mono;     // display plot in black and white
   private ArrayList file;   // list of files to be plotted

   /* Method      - Arguments - Default Constructor
    * Parameters  - NULL
    * Returns     - NULL
    * Description - sets all the flags to false
    */
   public Arguments()
     {
	one     = false;
	help    = false;
	version = false;
	file    = new ArrayList();
     }
   
   /* Method      - isOne
    * Parameters  - NULL
    * Returns     - one
    * Description - returns the value of the flag "one"
    */
   public boolean isOne() { return(one); }

   /* Method      - setOne
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "one"
    */
   private void setOne(boolean flag) { one = flag; }

   /* Method      - isMargin
    * Parameters  - NULL
    * Returns     - one
    * Description - returns the value of the flag "margin"
    */
   public boolean isMargin() { return(margin); }

   /* Method      - setMargin
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "margin"
    */
   private void setMargin(boolean flag) { margin = flag; }

   /* Method      - isHelp
    * Parameters  - NULL
    * Returns     - help
    * Description - returns the value of the flag "help"
    */
   public boolean isHelp() { return(help); }
   
   /* Method      - setHelp
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "help"
    */
   private void setHelp(boolean flag) { help = flag; }

   /* Method      - isVersion
    * Parameters  - NULL
    * Returns     - one
    * Description - returns the value of the flag "Version"
    */
   public boolean isVersion() { return(version); }

   /* Method      - setVersion
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "version"
    */
   private void setVersion(boolean flag) { version = flag; }

   /* Method      - isTile
    * Parameters  - NULL
    * Returns     - help
    * Description - returns the value of the flag "tile"
    */
   public boolean isTile() { return(tile); }

   /* Method      - setTile
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "tile"
    */
   private void setTile(boolean flag) { tile = flag; }

   /* Method      - isCascade
    * Parameters  - NULL
    * Returns     - help
    * Description - returns the value of the flag "cascade"
    */
   public boolean isCascade() { return(cascade); }

   /* Method      - setCascade
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "cascade"
    */
   private void setCascade(boolean flag) { cascade = flag; }
   
   /* Method      - isMono
    * Parameters  - NULL
    * Returns     - help
    * Description - returns the value of the flag "mono"
    */
   public boolean isMono() { return(mono); }

   /* Method      - setMono
    * Parameters  - boolean value the flag is to be set to
    * Returns     - NULL
    * Description - set the value of the flag "mono"
    */
   protected void setMono(boolean flag) { mono = flag; }
   
   /* Method      - getFileCount()
    * Parameters  - NULL
    * Returns     - integer count of the input files entered at the command-line.
    * Description - returns the count of the number of files in the ArrayList file
    */
   public int getFileCount()
     {
	return(file.size());
     }

   /* Method      - getFileName()
    * Parameters  - NULL
    * Returns     - name of the file at location "index" in the ArrayList
    * Description - looks up the ArrayList at location "index" and returns the
    * corresponding file name.
    */
   public String getFileName(int index)
     {
	return((String)file.get(index));
     }
   
   /* Method      - argsError
    * Parameters  - NULL
    * Returns     - NULL
    * Description - prints the "usage" message in the event of an argument error.
    */
   public void argsError()
     {
	System.out.println("");	
	System.out.println("usage: java jPlot [options] file [files]");
	System.out.println("------");
	System.out.println("");
	System.out.println("Options:");
	System.out.println("--------");	
	System.out.println("  -1              \t show each file one at a time, rather than all at once");
	System.out.println("  -e              \t leave an extra margin around the plot (useful if some of the labels are getting clipped off the edges)");
	System.out.println("  -t, -[-]tile    \t tile the plots so that they can all be seen when displayed, rather than being displayed one on top of the other");
	System.out.println("  -c, -[-]cascade \t cascade the plots");
	System.out.println("  -v, -[-]version \t print version information");
	System.out.println("  -[-]help        \t display this help screen");
	System.out.println("");
	System.out.println("Mouse Bindings:");
	System.out.println("---------------");	
	System.out.println("  left         \t\t draw rectangle to zoom in, click to zoom out");
	System.out.println("  SHIFT + left \t\t drag to scroll window");
	System.out.println("  CTRL  + left \t\t drag out a box showing dimensions");
	System.out.println("  right        \t\t cycle between files   (if the \'-1\' option is used)");
	System.out.println("  right        \t\t quit                  (if the \'-1\' option is NOT used)");
	System.out.println("");
	System.out.println("Color Options:");
	System.out.println("--------------");
	System.out.println("  White on Black \t the plotter would use a WHITE stroke on a BLACK background");
	System.out.println("  Black on White \t the plotter would use a BLACK stroke on a WHITE background");
	System.out.println("  Black on Gray  \t the plotter would use a BLACK stroke on a GRAY  background");
	System.out.println("  NOTE:          \t colors other than BLACK and WHITE would remain the same");
	System.out.println("");
	System.out.println("Save As... Options:");
	System.out.println("-------------------");
	System.out.println("Plots can be saved to files in the following formats:");
	System.out.println("");
	System.out.println("  .png     \t\t Portable Network Graphics (PNG) Format");
	System.out.println("  .a4.ps   \t\t A4 size postscript image");
	System.out.println("  .tex3.ps \t\t 3 plots per latex page size postscript image");
	System.out.println("  .lvs.ps  \t\t less vertical space postscript image (image occupies less vertical space)");
	System.out.println("");
	System.out.println("Buttons:");
	System.out.println("--------");
	System.out.println("  Refresh  \t\t refresh the screen");
	System.out.println("  Original \t\t zoom back to the original view of the plot");
	System.out.println("  Resync   \t\t re-read the data from the file and display a fresh plot");
	System.out.println("  Close    \t\t close the current window");
	System.out.println("  Exit     \t\t close all the windows and exit");
	System.out.println("");
	System.out.println("Accelerator Keys");
	System.out.println("----------------");
	System.out.println("  CTRL + S \t\t Save As...");
	System.out.println("  CTRL + C \t\t close current window");
	System.out.println("  CTRL + X \t\t close all windows and exit");
	System.out.println("  CTRL + R \t\t refresh the screen");
	System.out.println("  CTRL + L \t\t resync i.e re-read the data from the file and display a fresh plot");
	System.out.println("  CTRL + H \t\t display a help dialog");
	System.out.println("");	
 
     }

   /* Method      - showVersion
    * Parameters  - NULL
    * Returns     - NULL
    * Description - prints the version string
    */
   public void showVersion()
     {
	/* Output version */
	System.out.println(Constant.PROGRAM_NAME_LONG + "\n" +
			   "Version: " + Constant.VERSION_MAJOR + "." + Constant.VERSION_MINOR + "." + Constant.VERSION_BUGFIX + "\n" +
			   "Release Date: " + Constant.RELEASE_DATE + "\n" +
			   Constant.COPYRIGHT + "\n"
			   );
	/* Exit */
	System.exit(1);
     }
   
   /* Method      - parseArgs
    * Parameters  - command line arguments
    * Returns     - success status
    * Description - parses the command-line arguments and sets the appropriate flags
    */
   public boolean parseArgs(String[] argv)
     {
	char[] arg;        // holds a single argument as an array of characters 
	int fileCount = 0; // file counter
	
	for(int i = 0; i < argv.length; ++i)
	  {
	     /* Convert the string to an array of characters for per-character comparisons */
	     arg = argv[i].toCharArray();

	     if(arg[0] == '-')
	       {
		  if((argv[i].compareTo("-v")        == 0) ||
		     (argv[i].compareTo("-version")  == 0) ||
		     (argv[i].compareTo("--version") == 0)
		    )
		    {
		       /* Set version flag */
		       setVersion(true);
		       /* Show the version number and exit */
		       showVersion();
		    }
		  else if((argv[i].compareTo("-h")     == 0) ||
			  (argv[i].compareTo("-help")  == 0) ||
			  (argv[i].compareTo("--help") == 0)
			 )
		    {
		       /* Set the help flag */
		       setHelp(true);
		       /* Display the help screen */
		       argsError();
		       /* Exit */
		       System.exit(1);
		    }
		  else if((argv[i].compareTo("-t")     == 0) ||
			  (argv[i].compareTo("-tile")  == 0) ||
			  (argv[i].compareTo("--tile") == 0)
			 )
		    {
		       /* Set the tile flag */
		       setTile(true);
		    }
		  else if((argv[i].compareTo("-c")        == 0) ||
			  (argv[i].compareTo("-cascade")  == 0) ||
			  (argv[i].compareTo("--cascade") == 0)
			 )
		    {
		       /* Set the tile flag */
		       setCascade(true);
		    }
		  else if((argv[i].compareTo("-m")     == 0) ||
			  (argv[i].compareTo("-mono")  == 0) ||
			  (argv[i].compareTo("--mono") == 0)
			 )
		    {
		       /* Set the mono flag */
		       setMono(true);
		    }
		  else
		    {
		       for(int j = 1; j < arg.length; ++j)
			 {
			    switch(arg[j])
			      {
			       case '1' : setOne(true)     ; break;
			       case 't' : setTile(true)    ; break;
			       case 'c' : setCascade(true) ; break;
			       case 'm' : setMono(true)    ; break;
			       case 'e' : setMargin(true)  ; break;
			       case 'v' : setVersion(true) ; showVersion() ; break;
			       case 'h' : setHelp(true)    ; argsError()   ; break;
			       default  : System.out.println("jPlot: invalid option -- " + arg[j]); return(false);
			      }
			 }
		    }
	       }
	     else
	       {
		  /* Grab plotter filename */
		  file.add(new String(argv[i]));
	       }
	  } //End of for(i...
	
	/* Check if any files plotter files were specified, else return false */
	if(file.size() == 0)
	  {
	     System.out.println("jPlot: No files to plot");
	     return(false);
	  }
	else
	  {
	     return(true);
	  }
	
	
     } //End of method parseArgs()
   
   public void dumpArgs()
     {
	/* Debugging */
	System.out.println("1 : " + one);
	System.out.println("h : " + help);
	System.out.println("v : " + version);
	
	for(int i = 0; i < file.size(); i++)
	  System.out.println("file " + i + " : " + file.get(i));
     }
   
} //End of class Arguments 
