package jplot;
/*
 * This file is part of jPlot
 *
 * jPlot is a Java version of Tim Shepard's xplot which is distributed 
 * under the M.I.T. license (as below).
 *
 * Tim Shepard's original xplot is an amazing piece of software. It was
 * designed to be extraordinarily fast and efficient and as such, it has
 * only a very thin user interface. Those design goals, however, make
 * xplot extremely difficult to port to new architectures and also make it
 * difficult to use for the casual user. jPlot was designed to address
 * these limitations while keeping as much of the original efficiency as
 * possible. We thank Tim Shepard for his hard work on xplot and hope that
 * jPlot can live up to his high standards.
 *
 * jPlot was developed by Avinash Lakhiani at the Internetworking Research
 * Lab (IRG), Ohio University, as part of Dr. Shawn Ostermann's tcptrace
 * project. The main goal behind the development of jPlot was to make the 
 * plotting tool used with tcptrace more portable. As a result, jPlot
 * retains many of the original algorithms implemented in xplot, and as per
 * the terms and conditions of the M.I.T. license, continue to remain under
 * that license. The M.I.T license is as follows:
 *
 * --- M.I.T License Begin ---
 * 
 * Copyright 1992,1993 by the Massachusetts Institute of Technology.
 *                   All rights reserved.
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
 * OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
 * limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
 * MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
 * OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
 * PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 *
 * The name of the Massachusetts Institute of Technology or M.I.T. may
 * NOT be used in advertising or publicity pertaining to distribution of
 * the software.  Title to copyright in this software and any associated
 * documentation shall at all times remain with M.I.T., and USER agrees
 * to preserve same.
 * 
 * --- M.I.T. License End ---
 * 
 * jPlot also has a lot of added features that are not part of the original
 * xplot code. Those sections of code are distributed under the following
 * BSD License:
 *
 * --- BSD License Begin ---
 *
 * Copyright (c) 2002, 2003 by Avinash Lakhiani
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *	
 *   - Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 *   - Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution. 
 *   - Neither the name of Ohio University nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission. 
 *	   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * --- BSD License End ---
 * 
 * You should have received a file named "DISTINCTION" along with this
 * distribution, that clearly identifies all the algorithms that were
 * retained from the original xplot code, and remain under the M.I.T.
 * license. The rest of the code was written for jPlot, and falls under
 * the BSD license.
 * 
 * jPlot is free software; you can redistribute it and/or modify it under the
 * terms and conditions of both the above licenses.
 *
 * jPlot is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the above licenses for more details.
 *
 * Author:  Avinash Lakhiani
 *          Internetworking Research Group
 *          Ohio University
 *          Athens, OH
 *          avinash.lakhiani@ohiou.edu
 *          http://www.tcptrace.org/jPlot
 *
 */

/*
 * $Id: cUnsigned.java,v 1.2 2002/12/04 01:58:53 alakhian Exp $
 */ 

/* Import Packages */
import java.lang.Long;
import java.lang.Math;

/* Class       - cUnsigned
 * Description - this class is for the coordinates of type signed integer. It implements all
 * the necessary methods for manipulation of instances of this type.
 * Extends     - NULL
 * Implements  - Coord
 */
public class cUnsigned implements Coord
{
   long u;
   
   /* Method      - cUnsigned (Constructor)
    * Parameters  - NULL
    * Returns     - NULL
    * Description - this constructor is called when a coordinate of type signed integer is
    * created with no initial value. The value is then set to 0 by default.
    */
   public cUnsigned()
     {
	this.zero();
     }

   /* Method      - cUnsigned (Constructor)
    * Parameters  - the string from which the signed integer value is to be extracted.
    * Returns     - NULL
    * Description - this constructor is called when a coordinate of type signed integer is
    * read in from the file as a string.
    */
   public cUnsigned(String s)
     {
	this.parse(s);
     }

   /* Method      - cUnsigned (Constructor)
    * Parameters  - the object from which the signed integer value is to be taken.
    * Returns     - NULL
    * Description - this constructor is called when a coordinate of type signed integer is obtained
    * from another coordinate
    */
   public cUnsigned(Coord c)
     {
	cUnsigned k = (cUnsigned)c;
	this.u = k.u;
     }

   /* Method      - set
    * Parameters  - value the coordinate is to be set to.
    * Returns     - NULL
    * Description - sets the value of the coordinate (field d) to the given value.
    */
   public void set(double val)
     {
	this.u = (long)val;
     }
   
   /* Method      - get
    * Parameters  - NULL
    * Returns     - the value of the coordinate
    * Description - fetches the value of the calling coordinate and returns it.
    */
   public double get()
     {
	return(this.u);
     }
   
   /* Method      - unparse
    * Parameters  - NULL
    * Returns     - the coordinate as a string
    * Description - used to convert the calling coordinate into a string
    */
   public String unparse()
     {
	Long lObj = new Long(this.u);
	return(lObj.toString());
     }
   
   /* Method      - parse
    * Parameters  - the string from which the signed integer value is to be extracted.
    * Returns     - NULL
    * Description - extracts the signed integer value from the provided string, and sets
    * the member variable i.
    */
   public void parse(String s)
     {
	Long lObj = new Long(s);
	this.u = lObj.longValue();
     }

   /* Method      - zero
    * Parameters  - NULL
    * Returns     - NULL
    * Description - sets the coordinate to zero.
    */
   public void zero()
     {
	this.u = 0;
     }
   
   /* Method      - compareTo
    * Parameters  - the coordinate with which the calling coordinate is to be compared.
    * Returns     - result of comparison i.e. -1 for <, 0 for ==, 1 for >.
    * Description - performs the comparison of the coordinates and returns the results.
    */
   public int compareTo(Coord c)
     {
	cUnsigned k = (cUnsigned)c;
	if(this.u > k.u) return(1);
	else if(this.u < k.u) return(-1);
	else return(0);
     }

   /* Method      - add
    * Parameters  - the coordinate to add to the calling coordinate.
    * Returns     - NULL
    * Description - adds the value of the passed coordinate to the calling coordinate.
    */
   public void add(Coord c)
     {
	cUnsigned k = (cUnsigned)c;
	this.u += k.u;
     }
   
   /* Method      - subtract
    * Parameters  - the coordinate to subtract from the calling coordinate.
    * Returns     - NULL
    * Description - subtracts the value of the passed coordinate from the calling coordinate.
    */
   public void subtract(Coord c)
     {
	cUnsigned k = (cUnsigned)c;
	this.u -= k.u;
     }
   
   /* Method      - roundUp
    * Parameters  - the rounding range
    * Returns     - NULL
    * Description - rounds-up the calling coordinate.
    */
   public void roundUp(Coord c1, Coord c2)
     {
	cUnsigned k1 = (cUnsigned)c1;
	cUnsigned k2 = (cUnsigned)c2;
	
	if(k1.u % k2.u == 0)
	  {
	     this.u = k1.u;
	  }
	else
	  {
	     this.u = k1.u % k2.u;
	     if (this.u < 0)
	       {
		  this.u += k2.u;
	       }
	     this.u = k1.u + (k2.u - this.u);
	  }
     }

   /* Method      - roundDown
    * Parameters  - the rounding range
    * Returns     - NULL 
    * Description - rounds-down the calling coordinate.
    */
   public void roundDown(Coord c1, Coord c2)
     {   
	cUnsigned k1 = (cUnsigned)c1;
	cUnsigned k2 = (cUnsigned)c2;

	this.u = k1.u - (k1.u % k2.u);
     }
   
   /* Method      - tick
    * Parameters  - the level
    * Returns     - NULL
    * Description - sets the appropriate tick value. Basically sets the tick marks for
    * the axis. 
    * NOTES       - this algorithm is taken (as-it-is) from Tim Shepard's xplot.
    */
   public void tick(int level)
     {
	this.u = 1;

	while (level >= 3)
	  {
	     this.u *= 10;
	     level -= 3;
	  }
	
	switch (level)
	  {
	   case 2:
	     this.u *= 5;
	     level -= 2;
	     break;
	   case 1:
	     this.u *= 2;
	     level -= 1;
	     break;
	   case 0:
	     break;
	  }
     }
   
   /* Method      - subTick
    * Parameters  - the level
    * Returns     - NULL
    * Description - sets the appropriate subtick value. Basically sets the subtick marks for
    * the axis. 
    * NOTES       - this algorithm is taken (as-it-is) from Tim Shepard's xplot.
    */
   public int subTick(int level)
     {
	int r = level;
	
	if (level < 2) r = 0;
	else
	  switch (level % 3)
	    {
	     case 2:
	       r = level - 2;
	       break;
	     case 1:
	       r = level - 1;
	       break;
	     case 0:
	       r = level - 1;
	       break;
	    }
	
	return(r);
     }
   
   
   /* Method      - map
    * Parameters  - the boundary coordinates, and the drawing area size.
    * Returns     - the mapped double value of the calling coordinate.
    * Description - maps the calling coordinate to the cooresponding coordinate on the
    * viewport.
    */
   public double map(Coord c1, Coord c2, double n)
     {
	double r;
	cUnsigned k1 = (cUnsigned)c1;
	cUnsigned k2 = (cUnsigned)c2;

	r = (((double) ((double) this.u - (double) k1.u)) *
	     (((double) n) / ((double) (k2.u - k1.u))));

	return(r);
     }
   
   /* Method      - unmap 
    * Parameters  - the boundary coordinates, the size of the drawing area, and the
    * coordinate to be unmapped.
    * Returns     - NULL
    * Description - computes the window coordinates, and sets the calling coordinate to the
    * resultant value.
    */
   public void unmap(Coord c1, Coord c2, double n, double x)
     {
	cUnsigned k1 = (cUnsigned)c1;
	cUnsigned k2 = (cUnsigned)c2;

	this.u = (long)Math.rint(((double) k1.u) + (((double) x) * (((double) (k2.u - k1.u)) / (double) n)));
     }

   /* Method      - toString
    * Parameters  - NULL
    * Returns     - the string representation of the unsigned integer value
    * Description - gets the string representing the unsigned integer value, and returns it.
    */
   public String toString()
     {
	Long lObj = new Long(this.u);
	return(lObj.toString());
     }

} // End of class cUnsigned
