package brn.sim;

import java.io.Serializable;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.Map;

import brn.sim.DataManager.DataContribution;
import brn.sim.data.AbstractDiagramData;
import brn.sim.data.ForwardGraphData;
import brn.sim.data.LinkTableData;

/**
 * RMI Interface for the simulation driver.
 *
 * @author kurth
 */
public interface DriverRemote extends Remote {

  static final String CALLBACK_RMI_NAME = "DriverRemote.Callback";

  /** a diagram */
  static final int ITEM_TYPE_DATA = 0;
  static final int ITEM_TYPE_FWDGRAPH = 1;
  static final int ITEM_TYPE_LINKTABLE = 2;
  static final int ITEM_TYPE_ARPTABLE = 3;
  static final int ITEM_TYPE_CONFIG = 4;
  static final int ITEM_TYPE_TIMEBAR = 5;
  /** bunch of data to be displayed in properties view */
  static final int ITEM_TYPE_PROPERTIES = 6;
  static final int ITEM_TYPE_XPLOT = 7;

  /**
   * Container class for simulator information items.
   */
  public class Item implements Serializable {
    private static final long serialVersionUID = 1L;

    /** type of the item */
    public int type;

    /** id of the item in simulator */
    public int id;

    /** display name and path */
    public String[] path;

    public Item(int type, int id, String[] path) {
      this.type = type;
      this.id = id;
      this.path = path;
    }
  }


  // ////////////////////////////////////////////////
  // Data access
  //

  /**
   * @return the data object with the specified id.
   */
  AbstractDiagramData getData(int id) throws RemoteException;

  /**
   * @return the simulation configuration.
   */
  AbstractParams getConfig(int id) throws RemoteException;

  /**
   * @return the simulation configuration.
   */
  ForwardGraphData getForwardGraph(int id) throws RemoteException;

  /**
   * Gets the current link table content associated with the given key.
   * resulting collection has the type
   *  Collection<LinkTable.LinkInfo>, Collection<LinkTableHandlers.Node>.
   *
   * @param id the link table key.
   * @return the link table contens belonging to the given key.
   * @throws RemoteException
   */
  LinkTableData getLinkTable(int id) throws RemoteException;

  Map /*NetAddress -> MacAddress*/ getArpTable(int id) throws RemoteException;

  DataContribution getContribution(int id) throws RemoteException;

  // ////////////////////////////////////////////////
  // Simulation control
  //

  /**
   * Run the simulation until the given time is reached.
   * @param nextStop time of the next break
   */
  void simRun(long nextStop) throws RemoteException;

  /**
   * Pause simulation immediately
   */
  void simPause() throws RemoteException;

  /**
   * Stops the current simulation.
   */
  void simStop() throws RemoteException;


  /**
   * RMI interface for the gui connected to the driver.
   */
  public interface Callback extends Remote {

    // ////////////////////////////////////////////////
    // Data notifiers
    //

    void addSimulatorItems(Item[] items) throws RemoteException;


    // ////////////////////////////////////////////////
    // Control notifiers
    //

    /** Connects gui to driver */
    void connectDriver(DriverRemote driver) throws RemoteException;

    void simPaused(long time) throws RemoteException;

  }

}
