package brn.sim.builder;

import java.io.Serializable;

import jist.swans.Node;

/**
 * Interface for entity builders. Builders are capable of building complex
 * entities. They bridge the gap between parameters and simulation objects.
 * Building entities in a two-stage process: at first build is called for
 * all parts to build, afterwards all entities are hooked up.
 *
 * @author kurth
 */
public abstract class Builder {

  /** the associated provider */
  private Provider provider;

  /**
   * Parameter for entity builders.
   *
   * @author kurth
   */
  public static abstract class Params implements Cloneable, Serializable {
    public Object clone() throws CloneNotSupportedException {
      return super.clone();
    }
  }

  /**
   * Parameters, which specify the the builder to use.
   *
   * @author kurth
   */
  public interface ParamsWithBuilder {

    abstract Class builderClass();
  }

  /**
   * @return the provider
   */
  public Provider getProvider() {
    return provider;
  }

  /**
   * @param provider the provider to set
   */
  public void setProvider(Provider provider) {
    this.provider = provider;
  }

  /**
   * Returns the class of params on which the builder acts.
   *
   * @return the class of params on which the builder acts.
   */
  public abstract Class getParamClass();

  /**
   * First stage of entity building: creates the bare entity.
   *
   * @param params parameters for this entity, of class getParamClass()
   * @param node the current node.
   * @return the newly created entity.
   * @throws BuilderException thrown in the case of errors.
   */
  public abstract Object build(Params params, Node node) throws BuilderException;

  /**
   * Second stage of entity building: entity hookup.
   *
   * @param params parameters for this entity, of class getParamClass()
   * @param node the current node.
   * @param entity the newly created entity.
   * @throws BuilderException thrown in the case of errors.
   */
  public abstract void hookUp(Params params, Node node, Object entity) throws BuilderException;

  /**
   * Optional third build stage
   *
   * @param params parameters for this entity, of class getParamClass()
   * @param node the current node.
   * @param entity the newly created entity.
   * @throws BuilderException thrown in the case of errors.
   */
  public void postHookUp(Params params, Node node, Object entity)
      throws BuilderException {
    // pass
  }

//  /**
//   * Return the category for this builder.
//   * @return the category for this builder.
//   */
//  String getCategory();
//
//  /**
//   * Returns the type of the builder with its category.
//   * @return the type of the builder with its category.
//   */
//  int getType();

}
