package brn.swans.route;


import jist.runtime.JistAPI;
import jist.swans.mac.MacAddress;
import jist.swans.misc.Message;
import jist.swans.misc.MessageAnno;
import jist.swans.misc.Protocol;
import jist.swans.net.NetAddress;
import jist.swans.net.NetMessage;
import jist.swans.net.NetMessage.Ip;
import jist.swans.route.RouteInterface;

/**
 * McExOR routing entity interface.
 */
public interface RouteMcExORInterface extends RouteInterface, Protocol {

  /**
   * Initiates route discovery for the given destination. Route requests are
   * retransmitted if no reply is received after a timeout elapses.
   *
   * @param dest      the destination to which a route is being sought
   * @param requestId the ID number of this route discovery
   */
  void discoverRoute(NetAddress dest, short requestId);

  /**
   * Adds an entry into the gratuitous route reply table.
   *
   * @param originator the originator of the packet being replied to
   * @param lastHop    the last-hop address of the packet being replied to
   */
  void addRouteReplyEntry(NetAddress originator, NetAddress lastHop);

  /**
   * Removes an entry from the gratuitous route reply table.
   *
   * @param originator the originator of the packet being removed
   * @param lastHop    the last-hop address of the packet being removed
   */
  void deleteRouteReplyEntry(NetAddress originator, NetAddress lastHop);

  /**
   * Inserts an packet into the send buffer.
   *
   * @param msg the packet to insert into the send buffer
   */
  void insertBuffer(NetMessage.Ip msg, MessageAnno anno);

  /**
   * Removes an entry from the send buffer.
   *
   * @param packet the <code>BufferedPacket</code> to be removed from the send
   *               buffer
   */
  void deleteBuffer(Object bufferedPacket);

  /**
   * RouteExORFlood: sends out a new flooding packet
   */
  void runFloodingSendHook();

  
  public final static class Dlg implements RouteMcExORInterface, JistAPI.Proxiable {
    RouteMcExORInterface impl;
    
    public Dlg(RouteMcExORInterface impl) {
      this.impl = impl;
    }

    public RouteMcExORInterface getProxy() {
      return (RouteMcExORInterface) JistAPI.proxy(this, RouteMcExORInterface.class);
    }

    public void addRouteReplyEntry(NetAddress originator, NetAddress lastHop) {
      impl.addRouteReplyEntry(originator, lastHop);
    }

    public void deleteBuffer(Object bufferedPacket) {
      impl.deleteBuffer(bufferedPacket);
    }

    public void deleteRouteReplyEntry(NetAddress originator, NetAddress lastHop) {
      impl.deleteRouteReplyEntry(originator, lastHop);
    }

    public void discoverRoute(NetAddress dest, short requestId) {
      impl.discoverRoute(dest, requestId);
    }

    public void insertBuffer(Ip msg, MessageAnno anno) {
      impl.insertBuffer(msg, anno);
    }

    public void peek(NetMessage msg, MacAddress lastHop, MessageAnno anno) {
      impl.peek(msg, lastHop, anno);
    }

    public void receive(Message msg, NetAddress src, MacAddress lastHop, byte macId, NetAddress dst, byte priority, byte ttl, MessageAnno anno) {
      impl.receive(msg, src, lastHop, macId, dst, priority, ttl, anno);
    }

    public void runFloodingSendHook() {
      impl.runFloodingSendHook();
    }

    public void send(NetMessage msg, MessageAnno anno) {
      impl.send(msg, anno);
    }

    public void start() {
      impl.start();
    }

  }
}
