//////////////////////////////////////////////////
// JIST (Java In Simulation Time) Project
// Timestamp: <Constants.java Wed 2005/02/09 18:18:27 barr rimbase.rimonbarr.com>
//

// Copyright (C) 2005 by Cornell University
// All rights reserved.
// Refer to LICENSE for terms and conditions of use.

package jist.swans;

import java.security.SecureRandom;
import java.util.Random;

/**
 * SWANS constants.
 *
 * @author Rimon Barr &lt;barr+jist@cs.cornell.edu&rt;
 * @version $Id: Constants.java,v 1.62 2005/02/09 23:32:20 barr Exp $
 * @since SWANS1.0
 */
public final class Constants
{

  //////////////////////////////////////////////////
  // Randomness
  //

  /** Global random number generator. */
  public static Random random = new SecureRandom();

  /** random number generator exclusively for placement */
  public static Random placementRandom = new SecureRandom();

  //////////////////////////////////////////////////
  // Time
  //

  /** zero delay. */
  public static final long PROCESS_IMMEDIATELY = 0;
  /** smallest possible delay. */
  public static final int EPSILON_DELAY = 1;
  /** one nano-second in simulation time units. */
  public static final long NANO_SECOND  = 1;
  /** one micro-second in simulation time units. */
  public static final long MICRO_SECOND = 1000 * NANO_SECOND;
  /** one milli-second in simulation time units. */
  public static final long MILLI_SECOND = 1000 * MICRO_SECOND;
  /** one second in simulation time units. */
  public static final long SECOND       = 1000 * MILLI_SECOND;
  /** one minute in simulation time units. */
  public static final long MINUTE       = 60 * SECOND;
  /** one hour in simulation time units. */
  public static final long HOUR         = 60 * MINUTE;
  /** one day in simulation time units. */
  public static final long DAY          = 24 * HOUR;

  //////////////////////////////////////////////////
  // Nature
  //

  /** Boltzmann's constant (units: Joules/Kelvin). */
  public static final double BOLTZMANN = 1.3807e-23;
  /** Speed of light in a vacuum (units: meter/second). */
  public static final double SPEED_OF_LIGHT = 2.9979e8;
  /** Pre-computed natural logarithm of 10. */
  public static final double log10 = Math.log(10);

  // pathloss constants
  public static final int PATHLOSS_FREE_SPACE = 1;
  public static final int PATHLOSS_TWO_RAY = 2;
  public static final int PATHLOSS_SHADOWING = 3;
  public static final int PATHLOSS_DIST_SHADOWING = 4;

  public static final int FADING_NONE = 0;
  public static final int FADING_RICIAN = 1;
  public static final int FADING_RAYLEIGH = 2;

  public static final int PROPAGATION_DELAY_NONE = 0;
  public static final int PROPAGATION_DELAY_FIXED = 1;
  public static final int PROPAGATION_DELAY_DIST = 2;
  public static final int PROPAGATION_DELAY_DIST_NORMAL = 3;

  
  //////////////////////////////////////////////////
  // Field-related constants
  //

  // constants

  /** Default field boundary (units: sim distance, usually meters). */
  public static final float FIELD_BOUND_X = (float)200.0, FIELD_BOUND_Y = (float)200.0;

  /** node placement choice constant. */
  public static final int PLACEMENT_INVALID = -1;
  /** node placement choice constant. */
  public static final int PLACEMENT_RANDOM  = 1;
  /** node placement choice constant. */
  public static final int PLACEMENT_GRID    = 2;
  /** node placement choice constant. */
  public static final int PLACEMENT_STREET_RANDOM = 3;
  /** node placement choice constant. */
  public static final int PLACEMENT_MANUAL = 4;
  /** node placement choice constant. */
  public static final int PLACEMENT_POISSON = 5;
  /** node placement choice constant. */
  public static final int PLACEMENT_MAX     = 5;
  /** node placement choice constant. */
  public static final int PLACEMENT_PYRAMID = 6;
  /** node placement choice constant. */
  public static final int PLACEMENT_DEFAULT = PLACEMENT_RANDOM;

  /** node mobility choice constant. */
  public static final int MOBILITY_INVALID  = -1;
  /** node mobility choice constant. */
  public static final int MOBILITY_STATIC   = 1;
  /** node mobility choice constant. */
  public static final int MOBILITY_WAYPOINT = 2;
  /** node mobility choice constant. */
  public static final int MOBILITY_TELEPORT = 3;
  /** node mobility random walk. */
  public static final int MOBILITY_RANDOMWALK     = 4;
  /** node mobility choice constant. */
  public static final int MOBILITY_STRAW_SIMPLE    = 5;
  /** node mobility choice constant. */
  public static final int MOBILITY_STRAW_OD   = 6;
//  /** node mobility choice constant. */
//  public static final int MOBILITY_DIRECTION = 7;
  /** node mobility choice constant. */
  public static final int MOBILITY_BOUNDLESS_SIM_AREA = 8;
  /** node mobility choice constant. */
  public static final int MOBILITY_DEFAULT  = MOBILITY_STATIC;

  /** street mobility configuration constant. */
  public static final int MOBILITY_STREET_RANDOM = 1;
  /** street mobility configuration constant. */
  public static final int MOBILITY_STREET_FLOW = 2;

  /** spatial data structure choice constant. */
  public static final int SPATIAL_INVALID = -1;
  /** spatial data structure choice constant. */
  public static final int SPATIAL_LINEAR  = 0;
  /** spatial data structure choice constant. */
  public static final int SPATIAL_GRID    = 1;
  /** spatial data structure choice constant. */
  public static final int SPATIAL_HIER    = 2;
  /** spatial data structure choice constant. */
  public static final int SPATIAL_WRAP    = 16;

  /** node RF channel choice constant. */
  public static final int CHANNEL_PATTERN_INVALID     = -1;
  /** node RF channel choice constant: all nodes have the same rf channel. */
  public static final int CHANNEL_PATTERN_HOMOGENEOUS = 0;
  /** node RF channel choice constant: rf channels are assigned in a round robin manner. */
  public static final int CHANNEL_PATTERN_ROUNDROBIN  = 1;
  /** node RF channel choice constant: rf channels are assigned in a random manner. */
  public static final int CHANNEL_PATTERN_RANDOM      = 2;
  /** node RF channel choice constant: rf channels are assigned along string. */
  public static final int CHANNEL_PATTERN_STRING      = 3;
  /** node RF channel choice constant: rf channels are assigned with the given offset. */
  public static final int CHANNEL_PATTERN_OFFSET      = 4;
  /** node RF channel choice constant: rf channels are assigned vertically. */
  public static final int CHANNEL_PATTERN_VERTICAL    = 5;
  /** node RF channel choice constant: rf channels are assigned according to a poisson process. */
  public static final int CHANNEL_PATTERN_POISSON     = 6;
  /** node RF channel choice constant: .... */
  public static final int CHANNEL_PATTERN_P2          = 7;

  /** the number of different channels */
  public static final int CHANNEL_NUMBER_DEFAULT      = 1;

  //////////////////////////////////////////////////
  // packet constants
  //

  /** packet with zero wire size. */
  public static final int ZERO_WIRE_SIZE = Integer.MIN_VALUE;

  //////////////////////////////////////////////////
  // Radio-related constants
  //

  /**
   * Air propagation delay.
   */
  public static final long PROPAGATION = Constants.MICRO_SECOND;

  // radio modes

  /** Radio mode: sleeping. */
  public static final byte RADIO_MODE_SLEEP        = -1;
  /** Radio mode: idle, no signals. */
  public static final byte RADIO_MODE_IDLE         = 0;
  /** Radio mode: some signals above sensitivity. */
  public static final byte RADIO_MODE_SENSING      = 1;
  /** Radio mode: signal locked and receiving packet. */
  public static final byte RADIO_MODE_RECEIVING    = 2;
  /** Radio mode: transmitting packet. */
  public static final byte RADIO_MODE_TRANSMITTING = 3;

  // radio noise types
  /** Radio noise independent */
  public static final byte RADIO_NOISE_INDEP = 1;
  /** Radio noise additive */
  public static final byte RADIO_NOISE_ADDITIVE = 2;

  // timing constants

//  /** RX-TX switching delay. */
//  public static final long RADIO_TURNAROUND_TIME = 5 * MICRO_SECOND;
//  /** physical layer delay. */
//  public static final long RADIO_PHY_DELAY = RADIO_TURNAROUND_TIME;
  /** Constant used to specify the default "delay to the wire". */
  public static final int RADIO_NOUSER_DELAY = -1;

  /** packet is received without errors */
  public static final Byte RADIO_RECV_STATUS_OK    = new Byte((byte)0);
  /** packet is received with crc errors */
  public static final Byte RADIO_RECV_STATUS_CRC   = new Byte((byte)1);
  /** packet is received with phy errors */
  public static final Byte RADIO_RECV_STATUS_PHY   = new Byte((byte)2);
  /** packet was lost on the wireless way */
  public static final Byte RADIO_RECV_STATUS_LOST  = new Byte((byte)4);

  // defaults
  //public static final int BANDWIDTH_802_11 = (int)25e6; // 25 MHz

  /** Frequency basic unit: 1 MHz = 1*10e6 Hz */
  public static final long FREQUENCY_UNIT = 1000000L;

  /** 900 MHz frequency constant (units: Hz). */
  public static final long FREQUENCY_900MHZ = 900 * FREQUENCY_UNIT;

  /** 2.4 GHz frequency constant (units: Hz). */
  public static final long FREQUENCY_2_4GHZ = 2400 * FREQUENCY_UNIT;

//  public static final long FREQUENCY_2_4GHZ_DSSS = 2401 * FREQUENCY_UNIT;

  /** 5 GHz frequency constant (units: Hz). */
  public static final long FREQUENCY_5GHZ = 5000 * FREQUENCY_UNIT;

  /** Default radio frequency (units: Hz). */
  public static final long FREQUENCY_DEFAULT = FREQUENCY_2_4GHZ; // 2.4 GHz

  /** Frequency basic unit: 1 GHz = 1*10e9 Hz */
  public static final long INVALID_FREQUENCY = -1;

  /** Default radio channel (units: 1-n). */
  public static final int CHANNEL_DEFAULT = 1;

  /** Default radio channel (units: 1-n). */
  public static final int INVALID_CHANNEL = -1;

  // channel spacings
  /** Channel spacing in 802.11b */
  public static final double CHANNEL_SPACING_80211b = (int)5e6; // 5 MHz

  /** Channel spacing in 802.11g */
  public static final double CHANNEL_SPACING_80211g = (int)5e6; // 5 MHz

  /** Channel spacing in 802.11a */
  public static final double CHANNEL_SPACING_80211a = (int)20e6; // 20 MHz

  /** Default channel spacing. */
  public static final double CHANNEL_SPACING_DEFAULT = CHANNEL_SPACING_80211b; // 2.4 GHz

  // channel width

  /** Channel width in 802.11b */
  public static final double CHANNEL_WIDTH_80211b = (int)22e6; // 22 MHz

  /** Channel width in 802.11g
   * TODO why 16.25 and not 20?
   */
//  public static final double CHANNEL_WIDTH_80211g = (int)16.25e6; // 16.25 MHz
  public static final double CHANNEL_WIDTH_80211g = (int)20e6; // 16.25 MHz

  /** Channel width in 802.11a
   * TODO why 16.25 and not 20?
   */
//  public static final double CHANNEL_WIDTH_80211a = (int)16.25e6; // 16.25 MHz
  public static final double CHANNEL_WIDTH_80211a = (int)20e6; // 16.25 MHz

  /** Default channel width. */
  public static final double CHANNEL_WIDTH_DEFAULT = CHANNEL_WIDTH_80211b; // 2.4 GHz

  // channel number

  /** Number of orthogonal channels in 802.11b */
  public static final int ORTOGONONAL_CHANNEL_NUMBER_80211b = 3;

  /** Number of orthogonal channels in 802.11g */
  public static final int ORTOGONONAL_CHANNEL_NUMBER_80211g = 3;

  /** Number of orthogonal channels in 802.11a */
  public static final int ORTOGONONAL_CHANNEL_NUMBER_80211a = 12;

  /** Default threshold signal-to-noise ratio. was 10.0*/
  public static final double SNR_THRESHOLD_DEFAULT = 10.0;

  /**
   * Definition of bandwidths/bitrates (units: bit/s).
   */
  /** Invalid bandwidth */
  public static final int BANDWIDTH_INVALID = -1;

  public static final int BANDWIDTH_40Kbps   = (int)40e3;
  /** 1 Mbit/s */
  public static final int BANDWIDTH_1Mbps    = (int) 1e6;
  /** 2 Mbit/s */
  public static final int BANDWIDTH_2Mbps    = (int) 2e6;
  /** 5.5 Mbit/s */
  public static final int BANDWIDTH_5_5Mbps  = (int) 55e5;
  /** 11 Mbit/s */
  public static final int BANDWIDTH_11Mbps   = (int) 11e6;

  /** 6 Mbit/s */
  public static final int BANDWIDTH_6Mbps    = (int) 6e6;
  /** 9 Mbit/s */
  public static final int BANDWIDTH_9Mbps    = (int) 9e6;
  /** 12 Mbit/s */
  public static final int BANDWIDTH_12Mbps   = (int) 12e6;
  /** 18 Mbit/s */
  public static final int BANDWIDTH_18Mbps   = (int) 18e6;
  /** 24 Mbit/s */
  public static final int BANDWIDTH_24Mbps   = (int) 24e6;
  /** 36 Mbit/s */
  public static final int BANDWIDTH_36Mbps   = (int) 36e6;
  /** 48 Mbit/s */
  public static final int BANDWIDTH_48Mbps   = (int) 48e6;
  /** 54 Mbit/s */
  public static final int BANDWIDTH_54Mbps   = (int) 54e6;
  /** 54 Mbit/s */
  public static final int BANDWIDTH_72Mbps   = (int) 72e6;


  /** Default radio bandwidth (units: bits/second). */
  public static final int BANDWIDTH_DEFAULT = BANDWIDTH_1Mbps;

  /** All bitrates in 802.11a/OFDM in ascending order. */
  public static final int[] BITRATES_OFDM  = new int[] { BANDWIDTH_6Mbps,
    BANDWIDTH_9Mbps, BANDWIDTH_12Mbps, BANDWIDTH_18Mbps, BANDWIDTH_24Mbps,
    BANDWIDTH_36Mbps, BANDWIDTH_48Mbps, BANDWIDTH_54Mbps };

  /** All bitrates in 802.11b/DSSS in ascending order. */
  public static final int[] BITRATES_DSSS = new int[] { BANDWIDTH_1Mbps,
    BANDWIDTH_2Mbps, BANDWIDTH_5_5Mbps, BANDWIDTH_11Mbps, BANDWIDTH_40Kbps };

  /** All bitrates in 802.11g in ascending order. */
  public static final int[] BITRATES_ALL  = new int[] { BANDWIDTH_1Mbps,
    BANDWIDTH_2Mbps, BANDWIDTH_5_5Mbps, BANDWIDTH_6Mbps,
    BANDWIDTH_9Mbps, BANDWIDTH_11Mbps, BANDWIDTH_12Mbps, BANDWIDTH_18Mbps,
    BANDWIDTH_24Mbps, BANDWIDTH_36Mbps, BANDWIDTH_48Mbps, BANDWIDTH_54Mbps };

  /** Basic rates in 802.11 in ascending order. */
  public static final int[] BASIC_RATES_80211 = new int[]{ BANDWIDTH_1Mbps };

  /** Basic rates in 802.11a and 802.11g (pure) in ascending order. */
  public static final int[] BASIC_RATES_80211AG = new int[]{ BANDWIDTH_6Mbps,
    BANDWIDTH_12Mbps, BANDWIDTH_24Mbps };

  /** Basic rates in 802.11b in ascending order.
   *  See 802.11b, 18.2.6 for a hint to the rates contained in this set. */
  public static final int[] BASIC_RATES_80211B = new int[]{ BANDWIDTH_1Mbps,
    BANDWIDTH_2Mbps, BANDWIDTH_5_5Mbps, BANDWIDTH_11Mbps };

  /** Basic rates in 802.11bg in ascending order. */
  public static final int[] BASIC_RATES_80211BG = new int[]{ BANDWIDTH_1Mbps,
    BANDWIDTH_2Mbps, BANDWIDTH_5_5Mbps, BANDWIDTH_6Mbps, BANDWIDTH_11Mbps,
    BANDWIDTH_12Mbps, BANDWIDTH_24Mbps };

  // Rate control algorithms

  /** Constant rate (no rate change) */
  public static final int RCA_CONSTANT_RATE               = 0;
  /** Auto-rate fallback */
  public static final int RCA_AUTO_RATE_FALLBACK          = 1;
  /** Adaptive auto-rate fallback */
  public static final int RCA_ADAPTIVE_AUTO_RATE_FALLBACK = 2;
  /** SampleRate (as proposed by John C. Bicket) */
  public static final int RCA_SAMPLE_RATE                 = 3;
  /** OrRBAR derived from RBAR */
  public static final int RCA_OR_RBAR_MIN_SAFE_NB         = 41;
  public static final int RCA_OR_RBAR_MIN_RATE            = 42;

  /** Default transmission strength (units: dBm). */
  public static final double TRANSMIT_DEFAULT = 15.0;
  /** Default antenna gain (units: dB). */
  public static final double GAIN_DEFAULT = 0.0;

  /**
   * Default radio reception sensitivity (units: dBm).
   *
   * AZu:
   * Defines the receiver sensitivity value of the radio receiver for arriving packets.
   * Packets with a power less than the threshold are not sensed by the receiver.
   * Such packets don't change the receiver's radio status to busy and they are not detected by the
   * MAC through its physical sensing mechanism. Such packets cause interference.
   *
   * AZu: value of -95 taken from OPNET (old value was -91 (spec requires at least -95dBm))
   * */
  public static final double SENSITIVITY_DEFAULT = -95.0;

  /**
   * Definition of threshold values for signal decoding.
   */
  /** Radio reception threshold basic unit (dBm). */
  public static final double THRESHOLD_UNIT = 1.0;

  /** Default radio reception threshold (units: dBm). */
  public static final double THRESHOLD_DEFAULT = -81.0 * THRESHOLD_UNIT;
  /** Default temperature (units: degrees Kelvin). */
  public static final double TEMPERATURE_DEFAULT = 290.0;
  /** Default temperature noise factor. */
  public static final double TEMPERATURE_FACTOR_DEFAULT = 10.0;
  /** Default ambient noise (units: mW). */
  public static final double AMBIENT_NOISE_DEFAULT = 0.0;
  /** Default minimum propagated signal (units: dBm). */
  //public static final double PROPAGATION_LIMIT_DEFAULT = -111.0;
  public static final double PROPAGATION_LIMIT_DEFAULT = SENSITIVITY_DEFAULT;
  /** Default radio height (units: sim distance units, usually meters). */
  public static final double HEIGHT_DEFAULT = 1.5;


  //////////////////////////////////////////////////
  // Phy-related constants
  //

  // modulations

  /** Modulation DBPSK: differential binary phase shift keying.
   *  Used in 802.11, and 802.11b/g for 1 Mbit/s. */
  public static final short MODULATION_DBPSK = 10;

  /** Modulation DBPSK: differential quadrature phase shift keying.
   *  Used in 802.11, and 802.11b/g for 2 Mbit/s. */
  public static final short MODULATION_DQPSK = 11;

  /** Modulation CCK:   complementary code keying
   *  Used in 802.11b/g for 5.5, 11, and 22 Mbit/s. */
  public static final short MODULATION_CCK = 12;

  /** Modulation PBCC:  packet binary convolutional coding.
   *  OPTIONAL. Used in 802.11b/g for 5.5, 11, 22 and 33 Mbit/s. */
  public static final short MODULATION_PBCC = 13;

  /** Modulation BPSK:  binary phase shift keying.
   *  Used in 802.11a and 802.11g for 6 and 9 Mbit/s. */
  public static final short MODULATION_BPSK = 14;

  /** Modulation QPSK:  quadrature phase shift keying.
   *  Used in 802.11a and 802.11g for 12 and 18 Mbit/s. */
  public static final short MODULATION_QPSK = 15;

  /** Modulation QAM-16: quadrature amplitude modulation.
   *  Used in 802.11a and 802.11g for 24 and 36 Mbit/s. */
  public static final short MODULATION_QAM_16 = 16;

  /** Modulation QAM-64: quadrature amplitude modulation.
   *  Used in 802.11a and 802.11g for 48 and 54 Mbit/s. */
  public static final short MODULATION_QAM_64 = 17;


  ///////////////////////////////////////////////
  // constants for 802.11a/g
  //

  /**
   * Transmission time of OFDM PHY preamble in 802.11a.
   * Preamble consists of 10 short (.8 us) and 2 long (3.2us + 1.6us guard)
   * training sequences (symbols): 10 x 0.8us + 1.6us + 2 x 3.2us = 16us.
   */
  public static final long PREAMBLE_OFDM = 16 * Constants.MICRO_SECOND;

  /**
   * Transmission time of OFDM PHY PLCP Header (24 bits), always at 6Mbit/s.
   * This part of the header excludes the 16 service bits that are transmitted
   * as part of and at the same speed as the payload.
   */
  public static final long PLCP_HEADER_OFDM = 4 * Constants.MICRO_SECOND;

  /** OFDM PHY synchronization time. */
  public static final long SYNCHRONIZATION_OFDM = PREAMBLE_OFDM +
  PLCP_HEADER_OFDM;

  /**
   * SPEC: The slot time for the OFDM PHY shall be 9 ?s, which is the sum of the
   * RX-to-TX turnaround time, MAC processing delay, and CCA detect time (< 4 us).
   * The propagation delay shall be regarded as being included in the CCA detect time.
   */
  public static final long SLOT_TIME_OFDM = 9 * Constants.MICRO_SECOND;

  /** SIFS for OFDM PHY. */
  public static final long SIFS_OFDM_5GHZ = 16 * Constants.MICRO_SECOND;

  /** Transmission time for one symbol in OFDM PHY. (0.8us guard + 3.2us data = 4us) */
  public static final long SYMBOL_TX_TIME_OFDM = 4 * Constants.MICRO_SECOND;

  /** cyclic prefix in 802.11ag is .8us */
  public static final long CYCLIC_PREFIX_OFDM = 800 * Constants.NANO_SECOND;
  
  ///////////////////////////////////////////////
  // constants for 802.11b and 802.11g
  //

  /**
   * Length of DS PHY long preamble (802.11b/g).
   * 144 bits at 1 Mbit/s.
   */
  public static final long PREAMBLE_DSSS_LONG = 144 * Constants.MICRO_SECOND;
  public static final long PREAMBLE = PREAMBLE_DSSS_LONG;

  /**
   * Length of DS PHY short preamble (802.11b/g).
   * 72 bits at 1 Mbit/s.
   */
  public static final long PREAMBLE_DSSS_SHORT = 72 * Constants.MICRO_SECOND;

  /**
   * Length of DS PHY PLCP Header  (802.11b/g).
   * 48 bits at 1Mb/s.
   */
  public static final long PLCP_HEADER_DSSS_LONG = 48 * Constants.MICRO_SECOND;
//public static final long PLCP_HEADER = PLCP_HEADER_DSSS_LONG;

  /**
   * Length of DS PHY PLCP Header (802.11b/g).
   * 48 bits at 2Mb/s.
   */
  public static final long PLCP_HEADER_DSSS_SHORT = 24 * Constants.MICRO_SECOND;

  /**
   * DS PHY synchronization time with long pramble and PLCP header.
   */
  public static final long    SYNCHRONIZATION_DSSS_LONG     = PREAMBLE_DSSS_LONG
                                                                + PLCP_HEADER_DSSS_LONG;

  /**
   * DS PHY synchronization time with short pramble and PLCP header.
   */
  public static final long SYNCHRONIZATION_DSSS_SHORT = PREAMBLE_DSSS_SHORT
                                                                + PLCP_HEADER_DSSS_SHORT;

//public static final long SYNCHRONIZATION = PREAMBLE + PLCP_HEADER;

  /** SIFS for DSSS/CCK PHY. */
  public static final long SIFS_DSSS = 10 * Constants.MICRO_SECOND;

  /** RX/TX turnaround time (switching from reception to transmission) in 802.11a.*/
  public static final long RX_TX_TURNAROUND__802_11a = 2 * Constants.MICRO_SECOND;
  /** RX/TX turnaround time (switching from reception to transmission) in 802.11(b/g).*/
  public static final long RX_TX_TURNAROUND__802_11bg = 5 * Constants.MICRO_SECOND;

  /** TX/RX turnaround time (switching from transmission to reception) in 802.11(b/g).
   *  See 802.11[b|g]-1999.pdf, sections 15.4.6.6, 18.4.6.9, and 19.8.4: < 10 us. */
//  public static final long TX_RX_TURNAROUND__802_11bg = 9 * Constants.MICRO_SECOND;
  /** TX/RX turnaround time for all 802.11 standards (a,b,g). */
  public static final long TX_RX_TURNAROUND = 9 * Constants.MICRO_SECOND;

  /** Slot time for DSSS/CCK PHY. */
  public static final long SLOT_TIME_DSSS = 20 * Constants.MICRO_SECOND;



  ///////////////////////////////////////////////
  // additional constants for 802.11g
  //
  // Most constants for OFDM are the same as in 802.11a.
  // All constants for DSSS are the same as in 802.11b.
  //
  // ERP: Extendended Rate PHY
  //

  /**
   * SIFS for ERP-OFDM.
   */
  public static final long SIFS_OFDM__2_4_GHZ = 10 * Constants.MICRO_SECOND;

  /**
   * Signal extention time according to 802.11g spec.
   * Ensures several things: decoding of convolutional code, correct update of NAV
   * vector, etc.
   * Every ERP_OFDM packet transmission is followed by this idle time.
   *
   * TODO implement !!
   */
  public static final long SIGNAL_EXTENSION = 6 * Constants.MICRO_SECOND;

  /**
   * Minimum collision window (for backoff).
   */
  public static final short CW_MIN_DSSS = 31;

  /**
   * Minimum collision window (for backoff).
   */
  public static final short CW_MIN_OFDM = 15;

  /**
   * Maximum collision window (for backoff).
   */
  public static final short CW_MAX = 1023;

  //////////////////////////////////////////////////
  // Mac-related constants
  //

  // defaults

  /** Default mac promiscuous mode. */
  public static final boolean MAC_PROMISCUOUS_DEFAULT = false;
  /** link layer delay. */
  public static final long LINK_DELAY = MICRO_SECOND;

  /** Invalid MAC number. */
  public static final short MAC_INVALID        = -1;
  /** Simple MAC number. */
  public static final short MAC_DUMB        = 1;

  /** Constant to indicate an 802.11 MAC layer. */
  public static final short MAC_802_11 = 2;
  /** Constant to indicate an 802.11a MAC layer. */
  public static final short MAC_802_11a = 3;
  /** Constant to indicate an 802.11b MAC layer (default: short header support). */
  public static final short MAC_802_11b = 4;
  /** Constant to indicate an 802.11b MAC layer (without short header support). */
  public static final short MAC_802_11b_LONG = 5;
  /** Constant to indicate an 802.11g MAC layer in compatibility mode. */
  public static final short MAC_802_11bg = 6;
  /** Constant to indicate an 802.11g MAC layer in pure g mode (OFDM only). */
  public static final short MAC_802_11g_PURE = 7;
  /** Default MAC layer: 802.11b. */
  public static final short MAC_802_11_DEFAULT = MAC_802_11b;

  /** Constant to indicate a McExOR MAC (multi-channel + opportunistic) */
  public static final short MAC_MCExOR = 8;
  
  /** Constant to indicate a manually configured MAC */
  public static final short MAC_CUSTOM = 9;

  /** Retransmissions attempted for short packets (those without RTS). */
  public static byte MAC_RETRY_LIMIT_SHORT    = 7;
  /**
   * Threshold packet size to activate RTS. Default=2347. Broadcast packets
   * never use RTS. Set to zero to always use RTS.
   * See 802.11-1999.pdf, p. 483, element "dot11RTSThreshold".
   */
  public static int  MAC_THRESHOLD_RTS              = 2347;
  /**
   * Threshold packet size above which to fragment into several frames.
   * See 802.11-1999.pdf, p. 484, element "dot11FragmentationThreshold".
   */
  public static int  MAC_THRESHOLD_FRAGMENTATION    = 2346;
  /** Retransmissions attempted for long packets (those with RTS). */
  public static byte MAC_RETRY_LIMIT_LONG           = 4;

  /** */

  /** Ether type for ip */
  public static final short MAC_ETHERTYPE_IP = (short)0x0800;
  /** Ether type for brn */
  public static final short MAC_ETHERTYPE_BRN = (short)0x8086;
  
  public static final short MAC_ETHERTYPE_ARP = (short)0x0806;

  public static final short MAC_ETHERTYPE_UNKNOWN = (short)-1;

  //////////////////////////////////////////////////
  // Network-related constants
  //

  public static final int NET_QUEUE_NODROP = 0;
  public static final int NET_QUEUE_DROPTAIL = 1;

  public static final int NET_IPV4 = 0;
  public static final int NET_CLICK = 1;

  /** network layer loss model choice constant. */
  public static final int NET_LOSS_INVALID       = -1;
  /** network layer loss model choice constant. */
  public static final int NET_LOSS_NONE          = 0;
  /** network layer loss model choice constant. */
  public static final int NET_LOSS_UNIFORM       = 1;
  /** network layer loss model choice constant. */
  public static final int NET_LOSS_DEFAULT       = NET_LOSS_NONE;

  /** network packet priority level. */
  public static final byte NET_PRIORITY_INVALID  = -1;

  /** Priorities as they are known in 802.1D */
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_BESTEFFORT = 0;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_BACKGROUND = 1;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_UNDEFINED  = 2;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_EXCELLENTEFFORT = 3;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_CONTROLLEDLOAD = 4;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_VIDEO  = 5;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_VOICE  = 6;
  /** network packet priority level. */
  public static final byte NET_PRIORITY_D_NETWORKCONTROL  = 7;

  /** network packet priority level count. */
  public static final byte NET_PRIORITY_NUM      = 8;

  /** network interface constant. */
  public static final int NET_INTERFACE_INVALID = -1;
  /** network interface constant. */
  public static final int NET_INTERFACE_LOOPBACK = 0;
  /** network interface constant. */
  public static final int NET_INTERFACE_DEFAULT = 1;

  /** network layer delay. */
  public static final long NET_DELAY = MICRO_SECOND;
  /** network layer max jitter (uniform distributed jittering) */
  public static final long NET_JITTER = 10 * Constants.MICRO_SECOND;
  /** default time-to-live. */
  public static final byte TTL_DEFAULT = 64;

  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_INVALID        = -1;
  /** arp protocol */
  public static final short NET_PROTOCOL_ARP            = 10;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_TCP            = 6;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_UDP            = 17;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_OSPF           = 87;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_BELLMANFORD    = 520;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_FISHEYE        = 530;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_AODV           = 123;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_DSR            = 135;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_MCEXOR         = 140;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_LINK_PROBE     = 142;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_FLOODING       = 143;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_ODMRP          = 145;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_LAR1           = 110;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_ZRP            = 133;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_NDP_DEFAULT  = 1;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_IARP_DEFAULT = 2;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_BRP_DEFAULT  = 3;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_IERP_DEFAULT = 4;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_IARP_ZDP     = 5;
  /** zrp-subprotocol number. */
  public static final short NET_PROTOCOL_ZRP_BRP_FLOOD    = 6;

  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_NO_NEXT_HEADER = 59;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_HEARTBEAT      = 500;
  /** network level (IP) protocol number for no routing. */
  public static final short NET_PROTOCOL_DUMB           = 666;
  /** network level (IP) protocol number. */
  public static final short NET_PROTOCOL_MAX            = 999;

  //////////////////////////////////////////////////
  // Routing-related constants
  //

  //////////////////////////////////////////////////
  // Transport-related constants
  //

  /** transport layer delay. */
  public static final long TRANS_DELAY = MICRO_SECOND;
  /** socket delay. */
  public static final long TRANS_PROCESSING_DELAY = MICRO_SECOND;

  /** transport level (tcp/udp) protocol number. */
  public static final short TRANS_PROTOCOL_INVALID = -1;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_ECHO      =  7;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_FTP       = 21;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_TELNET    = 23;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_SMTP      = 25;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_TIME      = 37;
  /** transport level (tcp/udp) protocol number. */
  public static final short TCP_PROTOCOL_HTTP      = 80;


  /** TCP States. */
  public final class TCPSTATES
  {
    /**
     * TCP state: LISTEN - represents waiting for a connection request from
     * any remote TCP and port.
     */
    public static final int LISTEN = 800;
    /**
     * TCP state: SYN-SENT - represents waiting for a matching connection
     * request after having sent a connection request.
     */
    public static final int SYN_SENT = 801;
    /**
     * TCP state: SYN-RECEIVED - represents waiting for a confirming
     * connection request acknowledgment after having both received and sent a
     * connection request.
     */
    public static final int SYN_RECEIVED = 802;
    /**
     * TCP state: ESTABLISHED - represents an open connection, data received
     * can be delivered to the user.  The normal state for the data transfer
     * phase of the connection.
     */
    public static final int ESTABLISHED = 803;
    /**
     * TCP state: FIN-WAIT-1 - represents waiting for a connection
     * termination request from the remote TCP, or an acknowledgment of the
     * connection termination request previously sent.
     */
    public static final int FIN_WAIT_1 = 804;
    /**
     * TCP state: FIN-WAIT-2 - represents waiting for a connection
     * termination request from the remote TCP.
     */
    public static final int FIN_WAIT_2 = 805;
    /**
     * TCP state: CLOSE-WAIT - represents waiting for a connection termination
     * request from the local user.
     */
    public static final int CLOSE_WAIT = 806;
    /**
     * TCP state: CLOSING - represents waiting for a connection termination
     * request acknowledgment from the remote TCP.
     */
    public static final int CLOSING = 807;
    /**
     * TCP state: LAST-ACK - represents waiting for an acknowledgment of the
     * connection termination request previously sent to the remote TCP (which
     * includes an acknowledgment of its connection termination request).
     */
    public static final int LAST_ACK = 808;
    /**
     * TCP state: TIME-WAIT - represents waiting for enough time to pass to be
     * sure the remote TCP received the acknowledgment of its connection
     * termination request.
     */
    public static final int TIME_WAIT = 809;
    /**
     * TCP state: CLOSED - represents no connection state at all.
     */
    public static final int CLOSED = 810;

  } // class: TCPSTATES

} // class: Constants

