package jist.swans.rate;

import jist.swans.mac.MacAddress;
import jist.swans.misc.Message;
import jist.swans.misc.MessageAnno;
import jist.swans.phy.PhyMessage;

/**
 * Interface for all rate control algorithms.
 * 
 * @author Oliver Friedrich ofriedri(at)informatik(dot)hu-berlin(dot)de
 * 
 */
public interface RateControlAlgorithmIF {

	/**
	 * Report the result of a packet transmission.
	 * 
	 * @param nextHop
	 *            Next hop of the the current packet
	 * @param type
	 *            MacMessage type
	 * @param size
	 *            Size of the transmitted message in bytes
	 * @param bitrate
	 *            Bit-rate at which the packet was sent
	 * @param shortTries
	 *            If size <= RTS threshold: how many times this packet had to be
	 *            sent if size > RTS threshold: how many times the RTS packet
	 *            had to be sent
	 * @param longTries
	 *            If size <= RTS threshold: 0 if size > RTS threshold: how many
	 *            times this packet had to be sent
	 * @param status
	 *            Transmission status of the given packet
	 */
	public void reportPacketTx(MacAddress nextHop, int type, int size,
			int bitrate, int shortTries, int longTries, int status);

	/**
	 * Report the reception of a packet.
	 * 
	 * @param msg
	 *            Packet that was received.
	 */
	public void reportPacketRx(PhyMessage msg);

	/**
	 * Get the data rate from the bit-rate selection algorithm for the next data
	 * or management packet to send. Called before every transmission try.
	 * 
	 * @param anno
	 * @param packet
	 * 			  The next data packet to send.
	 * @param nextHop
	 *            The destination address of the message that will be sent next.
	 * @param frameLen
	 *            The length of the frame to transmit in bytes.
	 * 
	 * @return The data rate that should be used for the transmission of the
	 *         next packet (units: bit/s).
	 */
	public int getNextDataRate(Message packet, MessageAnno anno,
			Object nextHop, int frameLen, int triesBefore);

	/**
	 * Returns the bit-rate in the basic rate set, that is supposed to be used
	 * for transmission of the next control frame to the given sender.
	 * 
	 * @param nextHop
	 *            The destination address of the control frame to send.
	 * @param dataRate
	 *            The bit-rate at which a possible previous frame was sent to
	 *            this STA. Not all algorithms use this value.
	 * 
	 * @return The basic bit-rate to use for the next control packet.
	 */
	public int getControlRate(MacAddress nextHop, int dataRate);

}
