#!/usr/bin/env python


from optparse import OptionParser
from gnuradio import gr, gru, blks2, audio
from gnuradio import eng_notation
from gnuradio.gr import firdes
import os, struct, copy, sys

class AudioSink(gr.hier_block2):
    def __init__(self, options, outputfile):
        gr.hier_block2.__init__(self, "AudioSink",
        gr.io_signature(1, 1, gr.sizeof_gr_complex), # Input signature
        gr.io_signature(0, 0, 0))                    # Output signature

        options = copy.copy(options)    # make a copy so we can destructively modify

        if options.dsp:
            self.dst = audio.sink( options.dsp_sample_rate )
        else:
            self.dst = gr.wavfile_sink(outputfile, 2, options.wav_sample_rate, 16)

        self.c_to_iq = gr.complex_to_float()
        self.connect( self, self.c_to_iq)
        self.connect( (self.c_to_iq, 0), (self.dst, 0))
        self.connect( (self.c_to_iq, 1), (self.dst, 1))

    def add_options(parser):
        # output: dsp
        parser.add_option("-d", "--dsp", action="store_true", default=False, help="use soundcard instead of wav file [default: False]")
        parser.add_option("-r", "--dsp-sample-rate", type="int", default=48000, help="soundcard sample rate [default: %default Hz]")

        # output: wav
        parser.add_option("-w", "--wav-sample-rate", type="int", default=48000, help="wav-file sample rate [default: %default Hz]")

    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)


class AudioSource(gr.hier_block2):
    def __init__(self, options, inputfile):
        gr.hier_block2.__init__(self, "AudioSource",
        gr.io_signature(0, 0, 0),                    # Input signature
        gr.io_signature(1, 1, gr.sizeof_gr_complex)) # Output signature

        options = copy.copy(options)    # make a copy so we can destructively modify

        if options.dsp:
            self.src = audio.source(options.dsp_sample_rate, "", True)
        else:
            self.src = gr.wavfile_source( inputfile, False )

        self.iq_to_c = gr.float_to_complex()
        if options.dsp and options.wait:
            samples = options.dsp_sample_rate * options.wait
            self._head0 = gr.head(gr.sizeof_float, samples)
            self._head1 = gr.head(gr.sizeof_float, samples)
            self.connect( (self.src, 0), self._head0, (self.iq_to_c, 0) )
            self.connect( (self.src, 1), self._head1, (self.iq_to_c, 1) )
            if verbose: print "installed %d second head filter on dsp (%d samples at %d sps)" % (options.wait, samples, options.dsp_sample_rate)
        else:
            self.connect( (self.src, 0), (self.iq_to_c, 0) )
            self.connect( (self.src, 1), (self.iq_to_c, 1) )

        self.connect( self.iq_to_c, self)

    def add_options(parser):
        parser.add_option("-d", "--dsp", action="store_true", default=False, help="use soundcard instead of wav file [default: False]")
        parser.add_option("-r", "--dsp-sample-rate", type="int", default=48000, help="soundcard sample rate [default: %default Hz]")
        parser.add_option("-w", "--wait", type="int", default=None, help="when on dsp, accept this many seconds worth of audio or give up and exit. [default=%default]")

    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)


class AudioTransmitter(gr.hier_block2):
    def __init__(self, options):
        gr.hier_block2.__init__(self, "AudioTransmitter",
        gr.io_signature(1, 1, gr.sizeof_gr_complex), # Input signature
        gr.io_signature(1, 1, gr.sizeof_gr_complex)) # Output signature

        options = copy.copy(options)    # make a copy so we can destructively modify

        self.input = gr.mute_cc()
        self.connect( self, self.input)

        if options.carrier_frequency == 0:
            self.mixer = self.input
        else:
            self.mixer   = gr.multiply_vcc(1)
            self.carrier = gr.sig_source_c( options.carrier_sample_rate, gr.GR_SIN_WAVE, options.carrier_frequency, 1.0 )
            self.lowpass = gr.fir_filter_ccf(1, firdes.low_pass(1, 48000, 48000/(2*options.samples_per_symbol)+500, 500, firdes.WIN_HAMMING, 6.76))
            self.connect(self.input, self.lowpass, (self.mixer, 0) )
            self.connect(self.carrier,   (self.mixer, 1) )

        self.amp = gr.multiply_const_cc(1);
        self.amp.set_k(options.amp_amplitude)
        self.connect(self.mixer, self.amp, self)

        if options.debug_wavs:
            from myblks import debugwav
            self._dpassw = debugwav("tx_passband", options)
            self._dprefw = debugwav("tx_prefband", options)
            self._dbasew = debugwav("tx_baseband", options)
            self.connect(self.amp, self._dpassw)
            self.connect(self.lowpass, self._dbasew)
            self.connect(self.input, self._dprefw)

        if options.debug_files:
            self._dpassf = gr.file_sink(gr.sizeof_gr_complex*1, "debug_tx_passband.d_c")
            self._dpreff = gr.file_sink(gr.sizeof_gr_complex*1, "debug_tx_prefband.d_c")
            self._dbasef = gr.file_sink(gr.sizeof_gr_complex*1, "debug_tx_baseband.d_c")
            self.connect(self.amp, self._dpassf)
            self.connect(self.pkt_queue, self._dpreff)
            self.connect(self.lowpass, self._dbasef)

    def add_options(parser):
        parser.add_option("-z", "--carrier-sample-rate", type="int", default=48000, help="carrier sample rate [default: %default Hz]")
        parser.add_option("-c", "--carrier-frequency",   type="int", default=5000, help="carrier wave frequency [default: %default Hz]")
        parser.add_option("-a", "--amp-amplitude", type="float", default=0.1, help="amplitude on the amplifier [default=%default]")
        parser.add_option("-z", "--samples-per-symbol", type="int", default=32, help="samples per symbol [default=%default]")

        # output: dsp
        #parser.add_option("-d", "--dsp", action="store_true", default=False, help="use soundcard instead of wav file [default: False]")
        #parser.add_option("-r", "--dsp-sample-rate", type="int", default=48000, help="soundcard sample rate [default: %default Hz]")

        # output: wav
        parser.add_option("-w", "--wav-sample-rate", type="int", default=48000, help="wav-file sample rate [default: %default Hz]")

        parser.add_option("", "--debug-wavs", action="store_true", default=False, help="dump received passband and baseband signals to wav files [default: False]")
        parser.add_option("", "--debug-files", action="store_true", default=False, help="dump received passband and baseband signals to binary files [default: False]")


    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)


class AudioReceiver(gr.hier_block2):
    def __init__(self, options):
        gr.hier_block2.__init__(self, "AudioReceiver",
        gr.io_signature(1, 1, gr.sizeof_gr_complex), # Input signature
        gr.io_signature(1, 1, gr.sizeof_gr_complex)) # Output signature

        options = copy.copy(options)    # make a copy so we can destructively modify

        self.input = gr.mute_cc()
        #self.bandpass = gr.fir_filter_ccf(1, firdes.band_pass(1, 48000,
        #    options.carrier_frequency - 48000/(2*options.samples_per_symbol)+500,
        #    options.carrier_frequency +  48000/(2*options.samples_per_symbol)+500,
        #    500, firdes.WIN_HAMMING, 6.76))
        #self.input = self.bandpass
        self.connect(self, self.input)

        if options.carrier_frequency == 0:
            self.mixer = self.input
        else:
            self.carrier  = gr.sig_source_c( options.carrier_sample_rate, gr.GR_SIN_WAVE, - options.carrier_frequency, 1.0 )
            self.mixer    = gr.multiply_vcc(1)
            self.connect(self.input, (self.mixer, 0) )
            self.connect(self.carrier, (self.mixer, 1) )

        self.amp = gr.multiply_const_cc(1);
        self.amp.set_k(options.amp_amplitude)
        self.lowpass = gr.fir_filter_ccf(1, firdes.low_pass(1, 48000, \
          48000/(2*options.samples_per_symbol)+500, 500, firdes.WIN_HAMMING, 6.76))
        self.connect(self.mixer, self.lowpass, self.amp, self)
        #self.connect(self.mixer, self.amp, self)

        if options.debug_wavs:
            from myblks import debugwav
            self._dpass = debugwav("rx_passband", options)
            self._dbase = debugwav("rx_baseband", options)
            self._dbamp = debugwav("rx_baseamp", options)
            self.connect(self.input, self._dpass)
            self.connect(self.mixer, self._dbase)
            self.connect(self.amp,   self._dbamp)

        if options.debug_files:
            self._dpassf = gr.file_sink(gr.sizeof_gr_complex*1, "debug_rx_passband.d_c")
            self._dbasef = gr.file_sink(gr.sizeof_gr_complex*1, "debug_rx_baseband.d_c")
            self.connect(self.input, self._dpassf)
            self.connect(self.mixer, self._dbasef)

    def add_options(parser):
        parser.add_option("",   "--carrier-sample-rate", type="int", default=48000, help="carrier sample rate [default: %default Hz]")
        parser.add_option("-c", "--carrier-frequency",   type="int", default=5000, help="carrier wave frequency [default: %default Hz]")
        parser.add_option("-a", "--amp-amplitude", type="float", default=1.0, help="amplitude on the amplifier [default=%default]")

        #parser.add_option("-d", "--dsp", action="store_true", default=False, help="use soundcard instead of wav file [default: False]")
        #parser.add_option("-r", "--dsp-sample-rate", type="int", default=48000, help="soundcard sample rate [default: %default Hz]")
        #parser.add_option("-w", "--wait", type="int", default=None, help="when on dsp, accept this many seconds worth of audio or give up and exit. [default=%default]")

        parser.add_option("", "--debug-wavs", action="store_true", default=False, help="dump received passband and converted baseband signals to wav files [default: False]")
        parser.add_option("", "--debug-files", action="store_true", default=False, help="dump received passband and baseband signals to binary files [default: False]")
        parser.add_option("-w", "--wav-sample-rate", type="int", default=48000, help="wav-file sample rate [default: %default Hz]")


    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)
