#!/usr/bin/env python

import os, struct, re, time, os, signal

from optparse import OptionParser
from gnuradio import gr, blks2, audio

from gnuradio.gr import firdes

class PacketSender():
    def __init__(self, callback, options):
        self.callback = callback
        self.verbose = options.verbose
        self.pkt_overhead = 4
        self.size = options.size
        self.redundant_copies = options.redundant_copies

    def make_packet(self, data, pktno, correct_size):
        header = struct.pack('!HH', pktno, len(data))
        packet = header + data

        if len(packet) < correct_size:
            packet += '\x00' * (correct_size - len(packet))

        if len(packet) != correct_size:
            raise RuntimeError, "INTERNAL ERROR: packet size(%d!=%d) is incorrect" % (len(packet), correct_size)

        return packet

    def run(self, inputfile):
        carrier = self.make_packet("\xff" * (self.size - self.pkt_overhead), 0, self.size)

        for i in range(3):
            self.callback.send_pkt(carrier)

        file_header = "filename=%s; size=%d;" % (os.path.basename(inputfile), os.path.getsize(inputfile))

        if len(file_header) > self.size - self.pkt_overhead:
            raise SystemExit, "ERROR: packet size must be big enough to include file header (%d bytes in this case)" \
                % (len(file_header) + self.pkt_overhead)

        if self.verbose: print "pkt(1) -- %s" % file_header
        packet = self.make_packet(file_header, 1, self.size)
        for i in range(self.redundant_copies):
            self.callback.send_pkt(packet) # we send redundant_copies as a crappy kind of FEC

        pktno = 2
        f = open(inputfile)

        while True:
            data = f.read(self.size - self.pkt_overhead)
            if data:
                packet = self.make_packet(data, pktno, self.size)
                if self.verbose: print "pkt(%d)" % pktno
                for i in range(self.redundant_copies):
                    # we send redundant_copies as a crappy kind of FEC
                    self.callback.send_pkt(packet)
                pktno += 1

            else: break

        for i in range(3):
            self.callback.send_pkt(carrier)

    def add_options(parser):
        parser.add_option("-s", "--size", type="int", default=200, help="set packet size [default=%default Bytes]")
        parser.add_option("-f", "--redundant-copies", type="int", default=1, help="number of copies of each data packet to send (-f 1 for no-copies, -f 0 prevents sending any data) [default=%default]")

    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)


class PacketReceiver():
    def __init__(self, options):
        self.header_re = re.compile("filename=(.+?); size=([0-9]+);")
        self.infile = False
        self.lpktno = 0
        self.errno = 0
        self.crrno = 0
        self.filesz = 0
        self.f = 0
        self.exit_on_receive = options.exit_on_receive
        self.verbose = options.verbose

    def rx(self, ok, payload):
        if not ok: # ok is the result of a crc32 check in pkt_utils
            if self.verbose: print "-recv error (%d)-" % self.errno
            self.errno += 1
            return

        (pktno, size) = struct.unpack('!HH', payload[0:4])
        payload = payload[4:]

        if size > len(payload):
            print "WARNING: size of data part is bigger than data block (%d>%d)" % (size, len(payload))

        if pktno<=0:
            if self.verbose: print "-carrier packet (%d)-" % self.crrno
            self.crrno += 1
            return

        if self.infile:
            if pktno == self.lpktno:
                pass
            else:
                if self.verbose: print "pktno: %d; remaining: %d; this-sz: %d;" % \
                    (pktno, self.filesz, size)
                self.f.write(payload[0:size])
                self.filesz -= size
                self.lpktno = pktno
                if self.filesz <= 0:
                    if self.filesz < 0:
                        print "WARNING: received more file than expected... difference: %d" % self.filesz
                    self.infile = False
                    self.f.close()
                    if self.verbose:
                        dt  = (time.time() - self.ctime)
                        bps = self.ofilesz / dt
                        print "file.close(); %d/%f = %f bps" % (self.ofilesz, dt, bps)
                    if self.exit_on_receive:
                        if self.verbose: print "sys.exit() due to --exit-on-receive"
                        os.kill(os.getpid(), signal.SIGINT)
                        # sys.exit()
                        # rant:
                        # python *can't* exit with threads running.
                        # no language handles threads well, why do people use them?
        else:
            m = self.header_re.match(payload)
            if m:
                self.fname = "test_" + m.group(1)
                self.infile = True
                self.f = open(self.fname, "w")
                self.ofilesz = self.filesz = int(m.group(2))
                self.ctime = time.time()
                self.lpktno = 1
                if self.verbose: print "pktno: %d; fname: %s; fsize: %d; this-sz: %d;" % \
                    (pktno, self.fname, self.filesz, size)
            else:
                if self.lpktno == pktno:
                    pass
                else:
                    print "WARNING: received file chunk while not receiving a file pktno=%d; size=%d" % (pktno, size)

    def add_options(parser):
        parser.add_option("-e", "--exit-on-receive", action="store_true", default=False, help="exit after receiving a file? [default: %default]")

    # Make a static method to call before instantiation
    add_options = staticmethod(add_options)
