#!/usr/bin/env python

import random, time, struct, sys, math, os

from gnuradio import gr, blks2, audio
from gnuradio import eng_notation
from gnuradio.eng_option import eng_option
from gnuradio.gr import firdes
from optparse import OptionParser

from AudioFrontend import AudioTransmitter, AudioSink, AudioReceiver
from PacketTransceiver import PacketSender, PacketReceiver

class AwgnChannel(gr.hier_block2):
    def __init__(self, sample_rate, noise_voltage, frequency_offset, seed=False):
        gr.hier_block2.__init__(self, "AwgnChannel",
              gr.io_signature(1, 1, gr.sizeof_gr_complex), # Input signature
              gr.io_signature(1, 1, gr.sizeof_gr_complex)) # Output signature

        # Create the Gaussian noise source
        if not seed:
            self.noise = gr.noise_source_c(gr.GR_GAUSSIAN, noise_voltage)
        else:
            rseed = int(time.time())
            self.noise = gr.noise_source_c(gr.GR_GAUSSIAN, noise_voltage, rseed)

        self.adder =  gr.add_cc()

        # Create the frequency offset
        self.offset = gr.sig_source_c(1, gr.GR_SIN_WAVE,
                                      frequency_offset, 1.0, 0.0)
        self.mixer = gr.multiply_cc()

        # Connect the components
        self.connect(self, (self.mixer, 0))
        self.connect(self.offset, (self.mixer, 1))
        self.connect(self.mixer, (self.adder, 0))
        self.connect(self.noise, (self.adder, 1))
        self.connect(self.adder, self)


class TopBlockPassbandSim(gr.top_block):
    def __init__(self, options, callback):
        gr.top_block.__init__(self)

        self.audioTxFe = AudioTransmitter(options)
        #self.modulator = blks2.gmsk_mod(samples_per_symbol=options.samples_per_symbol)
        self.modulator = blks2.dbpsk_mod(samples_per_symbol=options.samples_per_symbol)
        self.pkt_queue1  = blks2.mod_pkts( modulator=self.modulator )

        self.audioRxFe = AudioReceiver(options)
        #self.demodulator = blks2.gmsk_demod(samples_per_symbol=options.samples_per_symbol)
        self.demodulator = blks2.dbpsk_demod(samples_per_symbol=options.samples_per_symbol)
        self.pkt_queue2  = blks2.demod_pkts( demodulator=self.demodulator, callback=callback, threshold=options.threshold )

        # Set up the parameters for the multipath fading channel:

        #vehicle_speed = 10 # the mobile speed (km/h)
        #carrier_freq = 5e3 # transmiter carrier frequance (Hz)
        #chan_rate = 48e3 # channel sample rate (Samples/sec)
        #chan_seed = -115        # Seeds the channel
        #chan_pwrs = (1, 1, 1, 1, 1 )    # The square root power of each resulting channel wave.
        #path_delays = (0, 1, 2, 3, 4)   # The delays for each path (us).
        #flag_indep = False      # Treats individual blocks as one continuous one.
        #flag_norm = False

        # This creates the fading channel based on the above parameters.
        #self.chan =  multipath_rayleigh_channel_cc(self, self.audioTxFe, vehicle_speed, carrier_freq,
        #     chan_rate, chan_seed, chan_pwrs, path_delays, flag_indep, flag_norm)

        SNR = 10.0**(options.snr/10.0)
        frequency_offset = options.frequency_offset
        timing_offset = options.timing_offset

        power_in_signal = abs(options.amp_amplitude)**2
        noise_power = power_in_signal/SNR
        noise_voltage = math.sqrt(noise_power)

        if options.taps:
            taps = [1.0, .2, 0.0, .1, .08, -.4, .12, -.2, 0, 0, 0, .3]
        else:
            taps = [1.0, 0.0]

        #self.channel = AwgnChannel(48000, noise_voltage, frequency_offset)
        self.channel = blks2.channel_model(noise_voltage, frequency_offset, timing_offset, taps)
        self.connect(self.pkt_queue1, self.audioTxFe, self.channel, self.audioRxFe, self.pkt_queue2)
        #self.connect(self.pkt_queue1, self.audioTxFe, self.audioRxFe, self.pkt_queue2)

    def send_pkt(self, msg=""):
        self.pkt_queue1.send_pkt(msg)

    def eof(self):
        self.send_pkt()
        self.pkt_queue1.send_pkt(eof=True)

    def main():
        usage="%prog: [options] inputfile"
        parser = OptionParser(usage, conflict_handler="resolve")

        parser.add_option("-q", "--quiet", action="store_false", dest="verbose", default=False, help="quiet? [default: False]")
        parser.add_option("-v", "--verbose", action="store_true", default=True, help="verbose? [default: %default]")
        parser.add_option("-t", "--threshold", type="int", default=12, help="detect frame header with up to threshold bits wrong [default=%default]")
        parser.add_option("", "--snr", type="float", default=30,
                           help="set the SNR of the channel in dB [default=%default]")
        parser.add_option("", "--frequency-offset", type="float", default=0,
                           help="set frequency offset introduced by channel")
        parser.add_option("", "--timing-offset", type="float", default=1,
                           help="A timing offset ratio to model clock difference")
        parser.add_option("-x", "--taps", action="store_true", default=False, help="taps? [default: %default]")

        PacketSender.add_options(parser)
        PacketReceiver.add_options(parser)
        txfe_grp = parser.add_option_group("Audio Transmitter Frontend")
        AudioTransmitter.add_options(txfe_grp)
        rxfe_grp = parser.add_option_group("Audio Receiver Frontend")
        AudioReceiver.add_options(rxfe_grp)

        (options, args) = parser.parse_args()

        inputfile = args[0]

        if not os.path.exists(inputfile):
            parser.print_help()
            raise SystemExit, "ERROR: input file must exist"

        try:
            prx = PacketReceiver(options)
            tb = TopBlockPassbandSim(options, prx.rx)
            ptx = PacketSender(tb, options)
            tb.start()

            ptx.run(inputfile)

            tb.eof()
            tb.wait()

        except KeyboardInterrupt:
            pass

    # Make a static method to call before instantiation
    main = staticmethod(main)

if __name__ == '__main__':
    TopBlockPassbandSim.main()
